% Model of a non-steady state non-isothermal plug flow reactor - Gas phase

% Main file

% For more information, see:

% A. De Visscher 
% Lecture Notes in Chemical Engineering Kinetics and Chemical Reactor Design
% Createspace (2013) 

% (C) Alex De Visscher (2013)

% ---------------------------

% data

clear all

tic
rhob = 450;          % Bed density of catalyst (kg/m3)
L = 4;               % Bed depth (m)
D = 0.05;            % Tube diameter (m)
Ntube = 300;         % Number of tubes
OMEGA = Ntube*(pi*D^2/4);  % Reactor cross sectional area
V = L*OMEGA;         % Reactor bed volume (m3)
v = 0:V/100:V;       % Reactor volume in 100 increments
deltaV = V/100;      % Volume increment
W = V*rhob;          % Catalyst mass (kg)
tspan = 0:15:150;    % We want output every 15 s up to 150 s
nEin = 70;           % Initial alkene flow rate (mol/s)
nHin = 10;           % Initial hydrogen flow rate (mol/s)
nAin = 0;            % Initial alkane flow rate (mol/s)
ntotin = nEin + nHin + nAin;

           % Coefficients for rate constants k2, KE, KH, and KA
Ak2 = 4.8e6;         % Rate constant k2 (mol/bar1.5/kg/s)
Ek2 = 110000;
BKE = 6.5e-6;        % Rate constant KE (1/bar)
HKE = 55000;
BKH = 7.0e-4;        % Rate constant KH (1/bar)
HKH = 20000;
BKA = 8.5e-5;        % Rate constant KA (1/bar)
HKA = 38000;

pin = 500000;        % Initial pressure (Pa)
ME = 0.084;          % Molar mass of E (kg/mol)
MH = 0.002;          % Molar mass of hydrogen
MA = 0.086;          % Molar mass of A
Tin = 663.15;        % Initial temperature (K)
Tc = 573.15;         % Cooling medium temperature (K)
dp = 0.006;          % Catalyst particle diameter (m)
mu = 2.5e-4;         % Gas viscosity (Pa s)
phi = 0.4;           % Bed porosity (-)

U = 60;              % Overall heat transfer coefficient (W/m2/K)
CpE = 246;           % Heat capacity of E (J/mol/K)
CpH = 29;            % Heat capacity of H2 (J/mol/K)
CpA = 266;           % Heat capacity of A (J/mol/K)
DH400 = -130000;     % Enthalply of reaction at 400 C
R = 8.314472;        % Ideal gas constant (J/mol K)
Cpcm = 1000;         % Specific heat of catalyst (J/kg K)

% y(1) to y(101) is concentration profile of A in reactor (mol/m3)
% y(102) to y(202) is concentration profile of H in reactor (mol/m3)
% y(203) to y(303) is concentration profile of E in reactor (mol/m3)
% y(304) to y(404) is temperature in reactor (K)

ctot = pin/(R*Tin);  % Total concentration at inlet (mol/m3)
usin = ntotin/(OMEGA*ctot);
y0(1) = ctot*nEin/ntotin;
y0(102) = ctot*nHin/ntotin;
y0(203) = ctot*nAin/ntotin;
y0(304) = Tin;
for j = 305:404
    y0(j) = Tc;
end

% Initial concentration profile: based on initial pressure profile

p = pin;
us = usin;
for j = 2:101
    rho = y0(j-1)*ME + y0(101+j-1)*MH + y0(202+j-1)+MA;
    G = rho*us;
    dpdV = -G/(OMEGA*rho*dp)*((1-phi)/phi^3)*(150*(1-phi)*mu/dp + 1.75*G);
    p = p + deltaV*dpdV;
    ctot = p/(R*y0(303+j));
    y0(j) = ctot;
    y0(101+j) = 0;
    y0(202+j) = 0;
end

% Call of differential equation solver

options = odeset('RelTol', 1e-6, 'AbsTol', 1e-8, 'InitialStep', 0.01);
[T,Y] = ode15s(@f,tspan,y0,options,Ak2,Ek2,BKE,HKE,BKH,HKH,BKA,HKA,ME,MH,MA,D,OMEGA,mu,Tc,dp,rhob,phi,U,CpE,CpH,CpA,DH400,Cpcm,usin,deltaV);

% Output

[s1,s2] = size(T);
c(1:101,1) = Y(s1,1:101);
c(1:101,2) = Y(s1,102:202);
c(1:101,3) = Y(s1,203:303);
Tempf(1:101,1) = Y(s1,304:404);
for j = 1:101
    ctotf(j,1) = c(j,1) + c(j,2) + c(j,3);
    pf(j,1) = ctotf(j,1)*R*Tempf(j,1);
    Vf(j,1) = (j-1)*V/100;
end

figure(1)
plot(Vf,c)
xlabel('Reactor volume (m3)');
ylabel('Concentration (mol/m3)');
figure(2)
plot(Vf,pf)
xlabel('Reactor volume (m3)');
ylabel('Pressure (Pa)');
figure(3)
plot(Vf,Tempf)
xlabel('Reactor volume (m3)');
ylabel('Temperature (K))');
V
Y
dpdVin = (pf(2,1) - pf(1,1))/deltaV;
rhoin = c(1,1)*ME + c(1,2)*MH + c(1,3)*MA;
Gin = (1/3.5)*(-150*(1-phi)*mu/dp + sqrt((150*(1-phi)*mu/dp)^2 - 7*dpdVin*rhoin*OMEGA*dp*phi^3/(1-phi)));
usin = Gin/rhoin;
nHin_actual = usin*OMEGA*c(1,2)
dpdVout = (pf(101,1) - pf(100,1))/deltaV;
rhoout = c(101,1)*ME + c(101,2)*MH + c(101,3)*MA;
Gout = (1/3.5)*(-150*(1-phi)*mu/dp + sqrt((150*(1-phi)*mu/dp)^2 - 7*dpdVout*rhoout*OMEGA*dp*phi^3/(1-phi)));
usout = Gout/rhoout;
conversion = (nHin_actual - usout*OMEGA*c(101,2))/nHin_actual
toc
